##--------------------------------------------------------------------------##
##  File:
##	$Id: Util.pm,v 1.1 2002/03/06 18:14:23 ehood Exp $
##  Description:
##	POD at end of file.
##--------------------------------------------------------------------------##
##  Copyright (C) 2002	Earl Hood <earl@earlhood.com>
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##  
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##  
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
##  02111-1307, USA
##--------------------------------------------------------------------------##

package MHArc::Util;

use Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(
  &ch_dir
  &cmd
  &mk_dir
  &run_prg
);

##--------------------------------------------------------------------------##

$ECHO_CMDS = 0;
$ECHO_ONLY = 0;

##--------------------------------------------------------------------------##

sub cmd {
  if ($ECHO_ONLY) {
    print "@_\n" if $ECHO_CMDS;
    return 0;   # bogus ok exit status
  }
  system @_;
}

sub ch_dir {
  my $dir       = shift;
  if ($ECHO_ONLY) {
    print "chdir $dir\n" if $ECHO_CMDS;
  } else {
    chdir $dir;
  }
}

sub run_prg {
  print "@_\n"  if $ECHO_CMDS;
  if (!$ECHO_ONLY) {
    my $wait = system @_;
    die qq/system @_ failed: $?\n/  if $wait;
  }
}

sub mk_dir {
  my $dir       = shift;
  my $mask      = shift;
  if ($ECHO_ONLY) {
    print "mkdir $dir\n"  if $ECHO_CMDS;
  } else {
    if (defined($mask)) {
      mkdir $dir, $mask;
    } else {
      mkdir $dir, 0777;
    }
  }
}


##--------------------------------------------------------------------------##
1;
__END__

=head1 NAME

MHArc::Util - General utilities for mail archiving system.

=head1 SYNOPSIS

  use MHArc::Util;

=head1 DESCRIPTION

This module contains a collection of utility routines.

=head1 VARIABLES

The following module variables can be set to affect the behavior
of the utility routines:

=over

=item C<$ECHO_CMDS>

If set to a true value, any routines that execute shell commands,
or external programs, will print the command to be executed to the
default filehandle.

The default value for C<$ECHO_CMDS> is 0.

=item C<$ECHO_ONLY>

If set to a true value, any routines that execute shell commands,
will only print the command to be executed to the default filehandle
if C<$ECHO_CMDS> is true.  The command itself will not be
executed at all.

The default value for C<$ECHO_ONLY> is 0.

=back

=head1 ROUTINES

By default, no routines are exported into the calling namespace.
Routines in this module can be imported by explicitly listing the
routines to import in the C<use> declaration:

  use MHArc::Util qw( run_prg );

The following routines are availale:

=over

=item C<ch_dir($dir)>

Change the current working directory.

=item C<cmd(LIST)>

Execute external program.  The return value is the wait(2) status
of the program invoked.

=item C<mk_dir($dir, $mask)>

Create specified directory.  C<$mask> is optional, and if
not specified, will default to C<0777>.

=item C<run_prg(LIST)>

Execute external program and terminate process if program returns
a non-zero status.

=back

=head1 VERSION

C<$Id>

=head1 AUTHOR

Earl Hood, earl@earlhood.com

