##--------------------------------------------------------------------------##
##  File:
##	$Id: CGI.pm,v 1.2 2002/09/18 17:23:29 ehood Exp $
##  Description:
##	POD at end of file.
##--------------------------------------------------------------------------##
##  Copyright (C) 2002	Earl Hood <earl@earlhood.com>
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##  
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##  
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
##  02111-1307, USA
##--------------------------------------------------------------------------##

package MHArc::CGI;

use Exporter;
@ISA = qw(Exporter);

@EXPORT_OK = qw(
  &parse_input
  &print_content_type
  &print_error
  &print_forbidden
  &print_input_error
  &print_location
  &print_not_found_error
  &print_script_error
);

##--------------------------------------------------------------------------##

BEGIN {
  $Debug = 0;
}

##--------------------------------------------------------------------------##

sub parse_input {
    my($method) = ($ENV{"REQUEST_METHOD"}) || 'GET';
    my($data);
    if ($method eq "GET") {
	$data = $ENV{"QUERY_STRING"} || "";
    } elsif ($method eq "POST") {
	read(STDIN, $data, $ENV{"CONTENT_LENGTH"});
    } else {
	warn qq/Unknown method: $method/;
	return undef;
    }

    my(@pairs, $name, $value);
    local $_;

    my $form = { };
    if ($data ne '') {
	@pairs = split(/&/, $data);
	foreach (@pairs) {
	    ($name, $value) = split(/=/);
	    $name = expandstr($name);
	    $value = expandstr($value);
	    $form->{$name} = $value;
	}
    }
    $form;
}

sub print_forbidden {
  print STDOUT 'Status: 403 Forbidden', "\r\n";
  print_content_type('text/plain');
  print STDOUT "Access Denied\n";
}

sub print_input_error {
  print STDOUT 'Status: 400 Bad Request', "\r\n";
  print_content_type('text/plain');
  print STDOUT "Input Error\n";
}

sub print_error {
  print_content_type('text/plain');
  print STDOUT "Script Error\n";
}

sub print_not_found_error {
  print STDOUT 'Status: 404 Not Found', "\r\n";
  print_content_type('text/plain');
  print STDOUT "Not Found\n";
}

sub print_location {
  print STDOUT 'Location: ', $_[0], "\r\n\r\n";
}

sub print_content_type {
    my($type) = shift;
    print STDOUT "Content-type: $type\r\n\r\n";
}

sub expandstr {
    my($str) = shift;
    $str =~ tr/+/ /;
    $str =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/ge;
    $str;
}

##--------------------------------------------------------------------------##
1;
__END__

=head1 NAME

MHArc::CGI - General CGI-related utilities for mail archiving system.

=head1 SYNOPSIS

  use MHArc::CGI;

=head1 DESCRIPTION

This module contains a collection of CGI-related utility routines used
by the various mharc CGI programs.

=head1 VARIABLES

The following module variables can be set to affect the behavior
of the utility routines:

=over

=item C<$Debug>

If set to a true value, routines will print out debugging information,
if appropriate.

=back

=head1 ROUTINES

By default, no routines are exported into the calling namespace.
Routines in this module can be imported by explicitly listing the
routines to import in the C<use> declaration:

  use MHArc::CGI qw( parse_input );

The following routines are availale:

=over

=item ...

=back

=head1 VERSION

C<$Id: CGI.pm,v 1.2 2002/09/18 17:23:29 ehood Exp $>

=head1 AUTHOR

Earl Hood, earl@earlhood.com

This module is part of the mharc archiving system and comes with
ABSOLUTELY NO WARRANTY and may be copied only under the terms of
the GNU General Public License, which may be found in the MHArc
distribution.

=cut

